#if NUNIT
#region Directives
//-----------------------***
using System;
using System.Collections;
using System.Data;
using System.Text;
using NUnit.Framework;
using gov.va.med.vbecs.UnitTests;

using TABLES = gov.va.med.vbecs.Common.VbecsTables;
using SPROCS = gov.va.med.vbecs.Common.VbecsStoredProcs;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;


using BOL = gov.va.med.vbecs.BOL;
using COMMON = gov.va.med.vbecs.Common;
using DAL = gov.va.med.vbecs.DAL;
using VAL = gov.va.med.vbecs.VAL;
using gov.va.med.vbecs.VAL.UnitTests;
//-----------------------***
using BROKEN = gov.va.med.vbecs.Common.VbecsBrokenRules.FrmShipmentIn;
#endregion
namespace gov.va.med.vbecs.BOL.UnitTests
{
	[TestFixture]
	public class BOL_IncomingShipment  : BaseTest
	{
		string _baseQuery = string.Concat( 
								"SELECT TOP 1 * FROM Shipment WHERE ShipmentDirection = 1 ",
								"AND RecordStatusCode = 'A' ",
								"AND DivisionCode = '", UnitTestConfig.TestDivisionCode, "' ",
								"AND ShipmentGuid IN (SELECT IncomingShipmentGuid FROM BloodUnitShipment WHERE IncomingShipmentGuid IS NOT NULL)" );

		private DataTable _dtShipment;
		private BOL.IncomingShipment _tstIS;
		private Guid _shipmentGuid;		

		[SetUp]
		protected void SetUp()
		{
			this.BaseSetUp();

			if (this.RefreshData)
			{
				this._dtShipment = UnitTestUtil.RunSQLGetDataTable(_baseQuery, false);

				// If there's no data then put some in
				if (this._dtShipment == null || this._dtShipment.Rows == null || this._dtShipment.Rows.Count == 0)
				{
					DataCreator.CreateBloodUnit(true);

					// Try again
					this._dtShipment = UnitTestUtil.RunSQLGetDataTable(_baseQuery, true);
				}
				
				this._shipmentGuid = (Guid)this._dtShipment.Rows[0][TABLES.Shipment.ShipmentGuid];

				this.RefreshData = false;
			}
			
			this._tstIS = new BOL.IncomingShipment(this._shipmentGuid);
			this._tstIS.SetInitialBrokenRules(BROKEN.FormClassName, BROKEN.RuleSets.Shipment);
		}

		[Test]
		public void GetIncomingShipments_Pass()
		{
			System.Collections.ArrayList alTest = BOL.IncomingShipment.GetIncomingShipments(this._tstIS.InvoiceNumber, this._tstIS.LocalSupplierGuid);
			Assert.IsTrue(alTest.Count > 0);
		}

		[Test]
		[ExpectedException(typeof(ArgumentNullException))]
		public void GetIncomingShipments_Fail()
		{
			BOL.IncomingShipment.GetIncomingShipments(null, Guid.NewGuid());
		}

		[Test]
		public void GetIncomingShipments_String_Guid_DateTime_Pass()
		{
			System.Collections.ArrayList alTest = BOL.IncomingShipment.GetIncomingShipments(this._tstIS.InvoiceNumber, this._tstIS.LocalSupplierGuid, this._tstIS.ShipmentDate);
			Assert.IsTrue(alTest.Count > 0);
		}

		[Test]
		[ExpectedException(typeof(ArgumentNullException))]
		public void GetIncomingShipments_String_Guid_DateTime_Fail()
		{
			BOL.IncomingShipment.GetIncomingShipments(null, Guid.NewGuid(), System.DateTime.Now);
		}

		[Test]
		public void Ctor_String_Guid_Guid_Date_Pass()
		{
			BOL.IncomingShipment tstIS = new BOL.IncomingShipment(this._tstIS.InvoiceNumber, this._tstIS.LocalSupplierGuid, this._tstIS.ProductShipperGuid, this._tstIS.ShipmentDate);

			Assert.AreEqual(this._tstIS.InvoiceNumber, tstIS.InvoiceNumber, "InvoiceNumber");
			Assert.AreEqual(this._tstIS.LocalSupplierGuid, tstIS.LocalSupplierGuid , "LocalSupplierGUid");
			Assert.AreEqual(this._tstIS.ProductShipperGuid, tstIS.ProductShipperGuid, "ProductShipperGuid");
			//Omitting DateCheck because of TimeZone garbage
		}

		[Test]
		[ExpectedException(typeof(ArgumentNullException))]
		public void Ctor_String_Guid_Guid_Date_Fail()
		{
			BOL.IncomingShipment tstIS = new BOL.IncomingShipment(null, Guid.Empty, Guid.Empty, System.DateTime.Now);
		}

		[Test]
		public void Ctor_DataRow_Pass()
		{
			BOL.IncomingShipment tstIS = new BOL.IncomingShipment(this._dtShipment.Rows[0]);

			Assert.AreEqual(this._tstIS.InvoiceNumber, tstIS.InvoiceNumber , "InvoiceNumber");
			Assert.AreEqual(this._tstIS.LocalSupplierGuid, tstIS.LocalSupplierGuid, "LocalSupplierGUid");
			Assert.AreEqual(this._tstIS.ProductShipperGuid, tstIS.ProductShipperGuid, "ProductShipperGuid");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void Ctor_DataRow_Fail()
		{
		}

		[Test]
		public void Ctor_Guid_Pass()
		{
			BOL.IncomingShipment tstIS = new BOL.IncomingShipment(this._tstIS.ShipmentGuid);

			Assert.AreEqual(this._tstIS.InvoiceNumber, tstIS.InvoiceNumber , "InvoiceNumber");
			Assert.AreEqual(this._tstIS.LocalSupplierGuid, tstIS.LocalSupplierGuid, "LocalSupplierGUid");
			Assert.AreEqual(this._tstIS.ProductShipperGuid, tstIS.ProductShipperGuid, "ProductShipperGuid");
			Assert.AreEqual(this._tstIS.ShipmentDate, tstIS.ShipmentDate, "ShipmentDate");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void Ctor_Guid_Fail()
		{
		}

		[Test]
		public void LocalSupplier_Pass()
		{
			BOL.Division tmpDiv = new Division(UnitTestConfig.TestDivisionCode);
			BOL.LocalSupplier tstLS = new LocalSupplier(tmpDiv.LocalSupplierGuid, false);

			this._tstIS.LocalSupplier = tstLS;
			Assert.AreEqual(this._tstIS.LocalSupplier , tstLS);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void LocalSupplier_Fail()
		{
		}

		[Test]
		public void ProductShipper_Pass()
		{
			Guid psGuid = DataCreator.GetRecordGuid("SELECT TOP 1 ProductShipperGuid FROM ProductShipper WHERE RecordStatusCode = 'A' AND DivisionCode = '" + UnitTestConfig.TestDivisionCode + "'");
			BOL.ProductShipper tstPS = new ProductShipper(psGuid);

			this._tstIS.ProductShipper = tstPS;
			Assert.AreEqual(this._tstIS.ProductShipper , tstPS);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void ProductShipper_Fail()
		{
		}

		[Test]
		public void ShipmentType_Pass()
		{
			Common.ShipmentType tstValue = Common.ShipmentType.ReturnToSupplier;

			this._tstIS.ShipmentType = tstValue;

			Assert.AreEqual(tstValue, this._tstIS.ShipmentType, "Property");
			Assert.AreEqual(false, this._tstIS.IsRuleBroken(BROKEN.ShipmentType), "BrokenRule");
		}

		[Test]
		public void ShipmentType_Fail()
		{
			this._tstIS.ShipmentType = Common.ShipmentType.Unknown;
			Assert.AreEqual(true, this._tstIS.IsRuleBroken(BROKEN.ShipmentType), "BrokenRule");
		}

		[Test]
		public void ShipmentGuid_Pass()
		{
			Guid tstValue = Guid.NewGuid();

			this._tstIS.ShipmentGuid = tstValue;

			Assert.AreEqual(tstValue, this._tstIS.ShipmentGuid);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void ShipmentGuid_Fail()
		{
		}

		[Test]
		public void ShipmentDirection_Pass()
		{
			Common.ShipmentDirection tstValue = Common.ShipmentDirection.OutgoingShipment;

			this._tstIS.ShipmentDirection = tstValue;

			Assert.AreEqual(tstValue, this._tstIS.ShipmentDirection, "Property");
			Assert.AreEqual(false, this._tstIS.IsRuleBroken(BROKEN.ShipmentDirection), "BrokenRule");
		}

		[Test]
		public void ShipmentDirection_Fail()
		{
			this._tstIS.ShipmentDirection = Common.ShipmentDirection.Unknown;
			Assert.AreEqual(true, this._tstIS.IsRuleBroken(BROKEN.ShipmentDirection), "BrokenRule");
		}

		[Test]
		public void InvoiceNumber_Pass()
		{
			string tstValue = "UT" + new Random().Next(111111,999999).ToString();

			this._tstIS.InvoiceNumber = tstValue;

			Assert.AreEqual(tstValue, this._tstIS.InvoiceNumber, "Property");
			Assert.AreEqual(false, this._tstIS.IsRuleBroken(BROKEN.InvoiceNumber), "BrokenRule");
		}

		[Test]
		public void InvoiceNumber_Fail()
		{
			this._tstIS.InvoiceNumber = string.Empty;
			Assert.AreEqual(true, this._tstIS.IsRuleBroken(BROKEN.InvoiceNumber), "BrokenRule");
		}

		[Test]
		public void ShipmentDate_Pass()
		{
			System.DateTime tstValue = System.DateTime.Now.AddYears(-1);

			this._tstIS.ShipmentDate = tstValue;

			Assert.AreEqual(tstValue, this._tstIS.ShipmentDate, "Property");
			Assert.AreEqual(false, this._tstIS.IsRuleBroken(BROKEN.ShipmentDate ), "BrokenRule");
		}

		[Test]
		public void ShipmentDate_Fail()
		{
			this._tstIS.ShipmentDate = System.DateTime.Now.AddYears(5);
			Assert.AreEqual(true, this._tstIS.IsRuleBroken(BROKEN.ShipmentDate ), "BrokenRule");
		}

		[Test]
		public void GetEmptyShipmentSaveTable_Pass()
		{
			DataTable dtTest = BOL.IncomingShipment.GetEmptyShipmentSaveTable();

			Assert.AreEqual(0, dtTest.Rows.Count, "Rows.Count");
			Assert.AreEqual(6, dtTest.Columns.Count, "Columns.Count");
			Assert.AreEqual(true , dtTest.Columns.Contains(TABLES.Shipment.ShipmentGuid), TABLES.Shipment.ShipmentGuid);
			Assert.AreEqual(true , dtTest.Columns.Contains(TABLES.Shipment.LocalSupplierGuid), TABLES.Shipment.LocalSupplierGuid);
			Assert.AreEqual(true , dtTest.Columns.Contains(TABLES.Shipment.InvoiceNumber), TABLES.Shipment.InvoiceNumber);
			Assert.AreEqual(true , dtTest.Columns.Contains(TABLES.Shipment.ShipmentDate), TABLES.Shipment.ShipmentDate);
			Assert.AreEqual(true , dtTest.Columns.Contains(TABLES.Shipment.ShipmentTypeCode), TABLES.Shipment.ShipmentTypeCode);
			Assert.AreEqual(true , dtTest.Columns.Contains(TABLES.Shipment.ShipmentDirection), TABLES.Shipment.ShipmentDirection);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetEmptyShipmentSaveTable_Fail()
		{
		}

		[Test]
		public void LoadDataRowFromThis_Pass()
		{
			DataRow drTest1 = this._dtShipment.NewRow();

			DataRow drTest2 = this._tstIS.LoadDataRowFromThis(drTest1);

			Assert.AreEqual(drTest1, drTest2);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void LoadDataRowFromThis_Fail()
		{
		}

		[Test]
		public void GetShipmentsWithAvailableUnitsPastFiveDays_Pass()
		{
			DataTable dtTest = BOL.IncomingShipment.GetShipmentsWithAvailableUnitsPastFiveDays();

			Assert.IsTrue(dtTest.Rows.Count >= 0);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetShipmentsWithAvailableUnitsPastFiveDays_Fail()
		{
		}

		[Test]
		public void GetShipment_Pass()
		{
			DataTable dtTest = BOL.IncomingShipment.GetShipment(this._tstIS.ShipmentGuid);

			Assert.AreEqual(1, dtTest.Rows.Count, "Rows.Count");
		}

		[Test]
		public void GetShipment_Fail()
		{
			DataTable dtTest = BOL.IncomingShipment.GetShipment(Guid.NewGuid());

			Assert.AreEqual(0, dtTest.Rows.Count, "Rows.Count");
		}

		[Test]
		public void GetShipmentByUnit_Pass()
		{
			string tmpSQL="SELECT TOP 1 * from BloodUnitShipment where RecordStatusCode = 'A'  and IncomingShipmentGuid is null ORDER BY LastUpdateDate ";
			DataTable dtTemp = UnitTestUtil.RunSQLGetDataTable(tmpSQL,true);

			Guid unitGuid = (Guid)dtTemp.Rows[0][TABLES.BloodUnitShipment.BloodUnitGuid];
			Guid osGuid = (Guid)dtTemp.Rows[0][TABLES.BloodUnitShipment.OutgoingShipmentGuid];
			
			DataTable dtTest = BOL.IncomingShipment.GetShipmentByUnit(unitGuid);

			Assert.IsTrue(dtTest.Rows.Count > 0);
			Assert.AreEqual(osGuid, (Guid)dtTest.Rows[0][TABLES.Shipment.ShipmentGuid], "ShipmentGuid");
		}

		[Test]
		public void GetShipmentByUnit_Fail()
		{
			DataTable dtTest = BOL.IncomingShipment.GetShipmentByUnit(Guid.NewGuid());

			Assert.AreEqual( 0, dtTest.Rows.Count );
		}

		[Test]
		public void GetShipmentType_Pass()
		{
			string tstValue = BOL.IncomingShipment.GetShipmentType(this._tstIS.ShipmentGuid);

			if (this._dtShipment.Rows[0].IsNull(TABLES.Shipment.ShipmentTypeCode))
			{
				Assert.AreEqual(string.Empty, tstValue);
			}
			else
			{
				Assert.AreEqual(this._tstIS.ShipmentTypeCode, tstValue);
			}
		}

		[Test]
		[ExpectedException(typeof(BOL.BusinessObjectException))]
		public void GetShipmentType_Fail()
		{
			BOL.IncomingShipment.GetShipmentType(Guid.NewGuid());
		}

		[Test]
		public void GetShipmentTypeByUnit_Pass()
		{
			string tmpSQL="SELECT TOP 1 * from BloodUnitShipment where RecordStatusCode = 'A'  and IncomingShipmentGuid is null ORDER BY LastUpdateDate ";
			DataTable dtTemp = UnitTestUtil.RunSQLGetDataTable(tmpSQL,true);

			Guid unitGuid = (Guid)dtTemp.Rows[0][TABLES.BloodUnitShipment.BloodUnitGuid];
			Guid shipGuid = (Guid)dtTemp.Rows[0][TABLES.BloodUnitShipment.OutgoingShipmentGuid];
			
			DataTable dtShip = UnitTestUtil.RunSQLGetDataTable("SELECT * FROM SHIPMENT WHERE ShipmentGuid = '" + shipGuid.ToString() + "'", true);
			string tstValue = BOL.IncomingShipment.GetShipmentTypeByUnit(unitGuid);


			if (dtShip.Rows[0].IsNull(TABLES.Shipment.ShipmentTypeCode))
			{
				Assert.AreEqual(string.Empty, tstValue);
			}
			else
			{
				Assert.AreEqual(dtShip.Rows[0][TABLES.Shipment.ShipmentTypeCode].ToString(), tstValue);
			}
		}

		[Test]
		public void GetShipmentTypeByUnit_Fail()
		{
			string tstValue = BOL.IncomingShipment.GetShipmentTypeByUnit(Guid.NewGuid());

			Assert.AreEqual(string.Empty, tstValue);
		}

		[Test]
		public void SaveIncomingShipment_Pass()
		{

			//This is essentially the same code from DataCreator.CreateBloodUnit() - only a wee more
			Guid bpGuid = DataCreator.GetRecordGuid("SELECT TOP 1 BloodProductGuid FROM BloodProduct");
			Guid unitGuid = Guid.NewGuid();
			Guid psGuid = Guid.NewGuid();
			Guid busGuid = Guid.NewGuid();

			BOL.BloodProduct tstBP = new BOL.BloodProduct(bpGuid);
			BOL.BloodUnit tmpUnit = new BOL.BloodUnit(unitGuid, bpGuid);
		
			BOL.Division tmpDiv = new BOL.Division(UnitTestConfig.TestDivisionCode);
			BOL.LocalSupplier tmpLS = new BOL.LocalSupplier(tmpDiv.LocalSupplierGuid, false);

			DataTable dtPS = DAL.ProductShipper.GetProductShipperByLocalSupplier(bpGuid, tmpDiv.LocalSupplierGuid);
			if (dtPS.Rows.Count == 0)
			{
				//No productShipper found, so create a new one
				BOL.ProductShipper tmpPS = new BOL.ProductShipper(psGuid, bpGuid);
				tmpPS.ProductCost = 1;
				tmpPS.LocalSupplierGuid = tmpLS.LocalSupplierGuid;
				tmpPS.ReturnCreditPercentage = 1;
				tmpPS.RecordStatusCode = Common.RecordStatusCode.Active;
				tstBP.ProductShippers.Add(tmpPS);
				tstBP.Save(Common.UpdateFunction.UnitTests);
			}
			else
			{
				psGuid = (Guid)dtPS.Rows[0][TABLES.ProductShipper.ProductShipperGuid];
			}

			BOL.IncomingShipment tmpShipment;

			DataTable dtS = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM Shipment WHERE InvoiceNumber = 'UnitTests' AND LocalSupplierGuid = '" + tmpLS.LocalSupplierGuid.ToString() + "'");
			if (dtS.Rows.Count == 0)
			{
				tmpShipment = new gov.va.med.vbecs.BOL.IncomingShipment("UnitTests", tmpLS.LocalSupplierGuid, psGuid,  System.DateTime.Now);
				tmpShipment.LocalSupplier = tmpLS;
			}
			else
			{
				tmpShipment = new BOL.IncomingShipment((Guid)dtS.Rows[0][TABLES.Shipment.ShipmentGuid]);
			}
			
			tmpShipment.DivisionCode = Common.LogonUser.LogonUserDivisionCode;

			//Set our BloodUnit properties
			tmpUnit.LabelTypeCode = "C";
			tmpUnit.UnitFdaRegistrationNumber = tmpLS.FDARegistrationNumber;

			string unitId = new Random().Next(11111,99999).ToString() + new Random().Next(11111,99999).ToString();
			tmpUnit.SetLabelUnitId(unitId, false);
			tmpUnit.SetEyeReadableUnitId(unitId, false);
			
			tmpUnit.BloodProductGuid = bpGuid;
			tmpUnit.DonationType = new BOL.DonationType(2);
			tmpUnit.OriginalVolume = 420;
			tmpUnit.DivisionCode = Common.LogonUser.LogonUserDivisionCode;
			tmpUnit.EntryMethodCode = "M";
			tmpUnit.RecordStatusCode = Common.RecordStatusCode.Active;
			tmpUnit.BloodUnitFinance.UnitCost = 98.76M;
			tmpUnit.BloodUnitFinance.SpecialTestingCost = 54.32M;
			tmpUnit.BloodUnitMedia.UnitExpirationDate = System.DateTime.Now.AddDays(7);
			tmpUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			tmpUnit.BloodUnitMedia.AboRh.RH = Common.RH.Positive;

			tmpUnit.Comments = "SaveIncomingShipment_Pass()";

			tmpUnit.BloodUnitStatus.BloodUnitStatusGuid = busGuid;
//			tmpUnit.BloodUnitStatus.UnitStatusCode = Common.Utility.GetUnitStatusCodeFromString("A");
			
			System.Collections.ArrayList emptyAL = new System.Collections.ArrayList();
			
			ArrayList antigens = new ArrayList();

			BOL.BloodUnitAntigen antigen = new BloodUnitAntigen();
			antigen.AntigenTypeId = (int)Common.Antigen.A;
			antigen.BloodUnitGuid = unitGuid;
			antigen.Positive = true;
			antigen.Tested = false;
			antigen.LastUpdateDate = DateTime.Now; 
			antigen.LastUpdateUser = Common.LogonUser.LogonUserName; 
			antigen.RowVersion = new byte[8];
			antigens.Add(antigen);

			BOL.BloodUnitAntigen antigenB = new BloodUnitAntigen();
			antigenB.AntigenTypeId = (int)Common.Antigen.B;
			antigenB.Positive = false; 
			antigenB.Tested = true;
			antigenB.BloodUnitGuid = unitGuid;
			antigenB.LastUpdateDate=DateTime.Now; 
			antigenB.LastUpdateUser=Common.LogonUser.LogonUserName;
			antigenB.RowVersion = new byte[8];
			antigens.Add(antigenB);

			string bloodUnit = tmpUnit.EyeReadableUnitId + "~" + tmpUnit.FullProductCode;

			DataCreator.CreateWorkloadProcess(Common.WorkloadProcessID.UnitLogin);
			tmpShipment.SaveIncomingShipment(tmpUnit, false , antigens, emptyAL, Common.UpdateFunction.UnitTests , bloodUnit,  Common.WorkloadProcessID.UnitLogin);

			Assert.IsTrue(true);
		}

		[Test]
		[ExpectedException(typeof(Common.DataAccessLayerException))]
		public void SaveIncomingShipment_Fail()
		{
			BOL.BloodUnit tmpUnit = new BOL.BloodUnit(Guid.NewGuid());
			tmpUnit.RowVersion = new byte[0];

			BOL.IncomingShipment tmpShipment = new BOL.IncomingShipment(Guid.NewGuid());
            BOL.Division tmpDiv = new BOL.Division(UnitTestConfig.TestDivisionCode);
            BOL.LocalSupplier tmpLS = new BOL.LocalSupplier(tmpDiv.LocalSupplierGuid, false);
		    tmpShipment.LocalSupplier = tmpLS;

			ArrayList emptyAL = new ArrayList();
			string bloodUnitList =  string.Empty;
			DataCreator.CreateWorkloadProcess(Common.WorkloadProcessID.UnitLogin);
			tmpShipment.SaveIncomingShipment(tmpUnit, true , emptyAL, emptyAL, Common.UpdateFunction.UnitTests , bloodUnitList, Common.WorkloadProcessID.UnitLogin);
		}

	}

}
#endif
